/*
 * Decompiled with CFR 0.152.
 */
package io.confluent.connect.storage.partitioner;

import io.confluent.connect.storage.common.SchemaGenerator;
import io.confluent.connect.storage.common.util.StringUtils;
import io.confluent.connect.storage.errors.PartitionException;
import io.confluent.connect.storage.partitioner.DefaultPartitioner;
import io.confluent.connect.storage.partitioner.TimestampExtractor;
import io.confluent.connect.storage.util.DataUtils;
import java.util.Date;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.regex.Pattern;
import org.apache.kafka.common.config.ConfigException;
import org.apache.kafka.common.utils.Time;
import org.apache.kafka.connect.connector.ConnectRecord;
import org.apache.kafka.connect.data.Schema;
import org.apache.kafka.connect.data.Struct;
import org.apache.kafka.connect.errors.ConnectException;
import org.apache.kafka.connect.sink.SinkRecord;
import org.joda.time.DateTime;
import org.joda.time.DateTimeZone;
import org.joda.time.format.DateTimeFormat;
import org.joda.time.format.DateTimeFormatter;
import org.joda.time.format.ISODateTimeFormat;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class TimeBasedPartitioner<T>
extends DefaultPartitioner<T> {
    private static final Logger log = LoggerFactory.getLogger(TimeBasedPartitioner.class);
    private static final String SCHEMA_GENERATOR_CLASS = "io.confluent.connect.storage.hive.schema.TimeBasedSchemaGenerator";
    private static final Pattern NUMERIC_TIMESTAMP_PATTERN = Pattern.compile("^-?[0-9]{1,19}$");
    private long partitionDurationMs;
    private String pathFormat;
    private DateTimeFormatter formatter;
    protected TimestampExtractor timestampExtractor;

    protected void init(long partitionDurationMs, String pathFormat, Locale locale, DateTimeZone timeZone, Map<String, Object> config) {
        this.delim = (String)config.get("directory.delim");
        this.partitionDurationMs = partitionDurationMs;
        this.pathFormat = pathFormat;
        try {
            this.formatter = TimeBasedPartitioner.getDateTimeFormatter(pathFormat, timeZone).withLocale(locale);
            this.timestampExtractor = this.newTimestampExtractor((String)config.get("timestamp.extractor"));
            this.timestampExtractor.configure(config);
        }
        catch (IllegalArgumentException e) {
            ConfigException ce = new ConfigException("path.format", (Object)pathFormat, e.getMessage());
            ce.initCause((Throwable)e);
            throw ce;
        }
    }

    private static DateTimeFormatter getDateTimeFormatter(String str, DateTimeZone timeZone) {
        return DateTimeFormat.forPattern((String)str).withZone(timeZone);
    }

    public static long getPartition(long timeGranularityMs, long timestamp, DateTimeZone timeZone) {
        long adjustedTimestamp = timeZone.convertUTCToLocal(timestamp);
        long partitionedTime = adjustedTimestamp / timeGranularityMs * timeGranularityMs;
        return timeZone.convertLocalToUTC(partitionedTime, false);
    }

    public long getPartitionDurationMs() {
        return this.partitionDurationMs;
    }

    public String getPathFormat() {
        return this.pathFormat;
    }

    public TimestampExtractor getTimestampExtractor() {
        return this.timestampExtractor;
    }

    @Override
    public void configure(Map<String, Object> config) {
        String localeString;
        super.configure(config);
        long partitionDurationMsProp = (Long)config.get("partition.duration.ms");
        if (partitionDurationMsProp < 0L) {
            throw new ConfigException("partition.duration.ms", (Object)partitionDurationMsProp, "Partition duration needs to be a positive.");
        }
        String pathFormat = (String)config.get("path.format");
        if (StringUtils.isBlank((String)pathFormat) || pathFormat.equals(this.delim)) {
            throw new ConfigException("path.format", (Object)pathFormat, "Path format cannot be empty.");
        }
        if (!StringUtils.isBlank((String)this.delim) && pathFormat.endsWith(this.delim)) {
            pathFormat = pathFormat.substring(0, pathFormat.length() - this.delim.length());
        }
        if (StringUtils.isBlank((String)(localeString = (String)config.get("locale")))) {
            throw new ConfigException("locale", (Object)localeString, "Locale cannot be empty.");
        }
        String timeZoneString = (String)config.get("timezone");
        if (StringUtils.isBlank((String)timeZoneString)) {
            throw new ConfigException("timezone", (Object)timeZoneString, "Timezone cannot be empty.");
        }
        Locale locale = new Locale(localeString);
        DateTimeZone timeZone = DateTimeZone.forID((String)timeZoneString);
        this.init(partitionDurationMsProp, pathFormat, locale, timeZone, config);
    }

    @Override
    public String encodePartition(SinkRecord sinkRecord, long nowInMillis) {
        Long timestamp = this.timestampExtractor.extract((ConnectRecord<?>)sinkRecord, nowInMillis);
        return this.encodedPartitionForTimestamp(sinkRecord, timestamp);
    }

    @Override
    public String encodePartition(SinkRecord sinkRecord) {
        Long timestamp = this.timestampExtractor.extract((ConnectRecord<?>)sinkRecord);
        return this.encodedPartitionForTimestamp(sinkRecord, timestamp);
    }

    private String encodedPartitionForTimestamp(SinkRecord sinkRecord, Long timestamp) {
        if (timestamp == null) {
            String msg = "Unable to determine timestamp using timestamp.extractor " + this.timestampExtractor.getClass().getName() + " for record: " + sinkRecord;
            log.error(msg);
            throw new PartitionException(msg);
        }
        DateTime bucket = new DateTime(TimeBasedPartitioner.getPartition(this.partitionDurationMs, timestamp, this.formatter.getZone()));
        return bucket.toString(this.formatter);
    }

    @Override
    public List<T> partitionFields() {
        if (this.partitionFields == null) {
            this.partitionFields = this.newSchemaGenerator(this.config).newPartitionFields(this.pathFormat);
        }
        return this.partitionFields;
    }

    @Override
    protected Class<? extends SchemaGenerator<T>> getSchemaGeneratorClass() throws ClassNotFoundException {
        return Class.forName(SCHEMA_GENERATOR_CLASS);
    }

    public TimestampExtractor newTimestampExtractor(String extractorClassName) {
        try {
            switch (extractorClassName) {
                case "Wallclock": 
                case "Record": 
                case "RecordField": {
                    extractorClassName = "io.confluent.connect.storage.partitioner.TimeBasedPartitioner$" + extractorClassName + "TimestampExtractor";
                    break;
                }
            }
            Class<?> klass = Class.forName(extractorClassName);
            if (!TimestampExtractor.class.isAssignableFrom(klass)) {
                throw new ConnectException("Class " + extractorClassName + " does not implement TimestampExtractor");
            }
            return (TimestampExtractor)klass.newInstance();
        }
        catch (ClassCastException | ClassNotFoundException | IllegalAccessException | InstantiationException e) {
            ConfigException ce = new ConfigException("Invalid timestamp extractor: " + extractorClassName);
            ce.initCause((Throwable)e);
            throw ce;
        }
    }

    public static class RecordFieldTimestampExtractor
    implements TimestampExtractor {
        private String fieldName;
        private DateTimeFormatter dateTime;

        @Override
        public void configure(Map<String, Object> config) {
            this.fieldName = (String)config.get("timestamp.field");
            this.dateTime = ISODateTimeFormat.dateTimeParser();
        }

        @Override
        public Long extract(ConnectRecord<?> record) {
            Object value = record.value();
            if (value instanceof Struct) {
                Struct struct = (Struct)value;
                Object timestampValue = DataUtils.getNestedFieldValue((Object)struct, (String)this.fieldName);
                Schema fieldSchema = DataUtils.getNestedField((Schema)record.valueSchema(), (String)this.fieldName).schema();
                if ("org.apache.kafka.connect.data.Timestamp".equals(fieldSchema.name())) {
                    return ((Date)timestampValue).getTime();
                }
                switch (fieldSchema.type()) {
                    case INT32: 
                    case INT64: {
                        return ((Number)timestampValue).longValue();
                    }
                    case STRING: {
                        return this.extractTimestampFromString((String)timestampValue);
                    }
                }
                log.error("Unsupported type '{}' for user-defined timestamp field.", (Object)fieldSchema.type().getName());
                throw new PartitionException("Error extracting timestamp from record field: " + this.fieldName);
            }
            if (value instanceof Map) {
                Map map = (Map)value;
                Object timestampValue = DataUtils.getNestedFieldValue((Object)map, (String)this.fieldName);
                if (timestampValue instanceof Number) {
                    return ((Number)timestampValue).longValue();
                }
                if (timestampValue instanceof String) {
                    return this.extractTimestampFromString((String)timestampValue);
                }
                if (timestampValue instanceof Date) {
                    return ((Date)timestampValue).getTime();
                }
                log.error("Unsupported type '{}' for user-defined timestamp field.", timestampValue.getClass());
                throw new PartitionException("Error extracting timestamp from record field: " + this.fieldName);
            }
            log.error("Value is not of Struct or Map type.");
            throw new PartitionException("Error encoding partition.");
        }

        private Long extractTimestampFromString(String timestampValue) {
            if (NUMERIC_TIMESTAMP_PATTERN.matcher(timestampValue).matches()) {
                try {
                    return Long.valueOf(timestampValue);
                }
                catch (NumberFormatException numberFormatException) {
                    // empty catch block
                }
            }
            return this.dateTime.parseMillis(timestampValue);
        }
    }

    public static class RecordTimestampExtractor
    implements TimestampExtractor {
        @Override
        public void configure(Map<String, Object> config) {
        }

        @Override
        public Long extract(ConnectRecord<?> record) {
            return record.timestamp();
        }
    }

    public static class WallclockTimestampExtractor
    implements TimestampExtractor {
        @Override
        public void configure(Map<String, Object> config) {
        }

        @Override
        public Long extract(ConnectRecord<?> record, long nowInMillis) {
            return nowInMillis;
        }

        @Override
        public Long extract(ConnectRecord<?> record) {
            return Time.SYSTEM.milliseconds();
        }
    }
}

