/*
 * Decompiled with CFR 0.152.
 */
package io.confluent.connect.s3.storage;

import com.amazonaws.AmazonClientException;
import com.amazonaws.event.ProgressEvent;
import com.amazonaws.event.ProgressListener;
import com.amazonaws.services.s3.AmazonS3;
import com.amazonaws.services.s3.model.AbortMultipartUploadRequest;
import com.amazonaws.services.s3.model.CannedAccessControlList;
import com.amazonaws.services.s3.model.CompleteMultipartUploadRequest;
import com.amazonaws.services.s3.model.InitiateMultipartUploadRequest;
import com.amazonaws.services.s3.model.PartETag;
import com.amazonaws.services.s3.model.SSEAlgorithm;
import com.amazonaws.services.s3.model.SSEAwsKeyManagementParams;
import com.amazonaws.services.s3.model.SSECustomerKey;
import com.amazonaws.services.s3.model.UploadPartRequest;
import io.confluent.connect.s3.S3SinkConnectorConfig;
import io.confluent.connect.s3.storage.ByteBuf;
import io.confluent.connect.s3.storage.CompressionType;
import io.confluent.connect.s3.storage.ElasticByteBuffer;
import io.confluent.connect.s3.storage.SimpleByteBuffer;
import io.confluent.connect.storage.common.util.StringUtils;
import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.ArrayList;
import java.util.Base64;
import java.util.List;
import java.util.function.Supplier;
import org.apache.commons.codec.digest.DigestUtils;
import org.apache.parquet.io.PositionOutputStream;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class S3OutputStream
extends PositionOutputStream {
    private static final Logger log = LoggerFactory.getLogger(S3OutputStream.class);
    private final AmazonS3 s3;
    private final String bucket;
    private final String key;
    private final String ssea;
    private final SSECustomerKey sseCustomerKey;
    private final String sseKmsKeyId;
    private final ProgressListener progressListener;
    private final int partSize;
    private final CannedAccessControlList cannedAcl;
    private boolean closed;
    private final ByteBuf buffer;
    private MultipartUpload multiPartUpload;
    private final CompressionType compressionType;
    private final int compressionLevel;
    private volatile OutputStream compressionFilter;
    private Long position;
    private final boolean enableDigest;

    public S3OutputStream(String key, S3SinkConnectorConfig conf, AmazonS3 s3) {
        this.s3 = s3;
        this.bucket = conf.getBucketName();
        this.key = key;
        this.ssea = conf.getSsea();
        String sseCustomerKeyConfig = conf.getSseCustomerKey();
        this.sseCustomerKey = SSEAlgorithm.AES256.toString().equalsIgnoreCase(this.ssea) && StringUtils.isNotBlank((String)sseCustomerKeyConfig) ? new SSECustomerKey(sseCustomerKeyConfig) : null;
        this.sseKmsKeyId = conf.getSseKmsKeyId();
        this.partSize = conf.getPartSize();
        this.cannedAcl = conf.getCannedAcl();
        this.closed = false;
        this.enableDigest = conf.isSendDigestEnabled();
        boolean elasticBufEnable = conf.getElasticBufferEnable();
        if (elasticBufEnable) {
            int elasticBufInitialCap = conf.getElasticBufferInitCap();
            this.buffer = new ElasticByteBuffer(this.partSize, elasticBufInitialCap);
        } else {
            this.buffer = new SimpleByteBuffer(this.partSize);
        }
        this.progressListener = new ConnectProgressListener();
        this.multiPartUpload = null;
        this.compressionType = conf.getCompressionType();
        this.compressionLevel = conf.getCompressionLevel();
        this.position = 0L;
        log.info("Create S3OutputStream for bucket '{}' key '{}'", (Object)this.bucket, (Object)key);
    }

    public void write(int b) throws IOException {
        this.buffer.put((byte)b);
        if (!this.buffer.hasRemaining()) {
            this.uploadPart();
        }
        Long l = this.position;
        Long l2 = this.position = Long.valueOf(this.position + 1L);
    }

    public void write(byte[] b, int off, int len) throws IOException {
        if (b == null) {
            throw new NullPointerException();
        }
        if (S3OutputStream.outOfRange(off, b.length) || len < 0 || S3OutputStream.outOfRange(off + len, b.length)) {
            throw new IndexOutOfBoundsException();
        }
        if (len == 0) {
            return;
        }
        if (this.buffer.remaining() <= len) {
            int firstPart = this.buffer.remaining();
            this.buffer.put(b, off, firstPart);
            this.position = this.position + (long)firstPart;
            this.uploadPart();
            this.write(b, off + firstPart, len - firstPart);
        } else {
            this.buffer.put(b, off, len);
            this.position = this.position + (long)len;
        }
    }

    private static boolean outOfRange(int off, int len) {
        return off < 0 || off > len;
    }

    private void uploadPart() throws IOException {
        this.uploadPart(this.partSize);
        this.buffer.clear();
    }

    private void uploadPart(int size) throws IOException {
        if (this.multiPartUpload == null) {
            log.debug("New multi-part upload for bucket '{}' key '{}'", (Object)this.bucket, (Object)this.key);
            this.multiPartUpload = this.newMultipartUpload();
        }
        try {
            this.multiPartUpload.uploadPart(new ByteArrayInputStream(this.buffer.array()), size);
        }
        catch (Exception e) {
            if (this.multiPartUpload != null) {
                this.multiPartUpload.abort();
                log.debug("Multipart upload aborted for bucket '{}' key '{}'.", (Object)this.bucket, (Object)this.key);
            }
            throw new IOException("Part upload failed: ", e);
        }
    }

    public void commit() throws IOException {
        if (this.closed) {
            log.warn("Tried to commit data for bucket '{}' key '{}' on a closed stream. Ignoring.", (Object)this.bucket, (Object)this.key);
            return;
        }
        try {
            this.compressionType.finalize(this.compressionFilter);
            if (this.buffer.hasRemaining()) {
                this.uploadPart(this.buffer.position());
            }
            this.multiPartUpload.complete();
            log.debug("Upload complete for bucket '{}' key '{}'", (Object)this.bucket, (Object)this.key);
        }
        catch (IOException e) {
            log.error("Multipart upload failed to complete for bucket '{}' key '{}'. Reason: {}", new Object[]{this.bucket, this.key, e.getMessage()});
            throw e;
        }
        finally {
            this.buffer.clear();
            this.multiPartUpload = null;
            this.internalClose();
        }
    }

    public void close() throws IOException {
        this.internalClose();
    }

    private void internalClose() throws IOException {
        if (this.closed) {
            return;
        }
        this.closed = true;
        if (this.multiPartUpload != null) {
            this.multiPartUpload.abort();
            log.debug("Multipart upload aborted for bucket '{}' key '{}'.", (Object)this.bucket, (Object)this.key);
        }
        super.close();
    }

    private MultipartUpload newMultipartUpload() throws IOException {
        InitiateMultipartUploadRequest initRequest = new InitiateMultipartUploadRequest(this.bucket, this.key).withCannedACL(this.cannedAcl);
        if (SSEAlgorithm.KMS.toString().equalsIgnoreCase(this.ssea) && StringUtils.isNotBlank((String)this.sseKmsKeyId)) {
            log.debug("Using KMS Key ID: {}", (Object)this.sseKmsKeyId);
            initRequest.setSSEAwsKeyManagementParams(new SSEAwsKeyManagementParams(this.sseKmsKeyId));
        } else if (this.sseCustomerKey != null) {
            log.debug("Using KMS Customer Key");
            initRequest.setSSECustomerKey(this.sseCustomerKey);
        }
        return S3OutputStream.handleAmazonExceptions(() -> new MultipartUpload(this.s3.initiateMultipartUpload(initRequest).getUploadId()));
    }

    private static <T> T handleAmazonExceptions(Supplier<T> supplier) throws IOException {
        try {
            return supplier.get();
        }
        catch (AmazonClientException e) {
            throw new IOException(e.getMessage(), e);
        }
    }

    public OutputStream wrapForCompression() {
        if (this.compressionFilter == null) {
            this.compressionFilter = this.compressionType.wrapForOutput((OutputStream)((Object)this), this.compressionLevel);
        }
        return this.compressionFilter;
    }

    public long getPos() {
        return this.position;
    }

    private static class ConnectProgressListener
    implements ProgressListener {
        private ConnectProgressListener() {
        }

        public void progressChanged(ProgressEvent progressEvent) {
            log.debug("Progress event: " + progressEvent);
        }
    }

    private class MultipartUpload {
        private final String uploadId;
        private final List<PartETag> partETags;

        public MultipartUpload(String uploadId) {
            this.uploadId = uploadId;
            this.partETags = new ArrayList<PartETag>();
            log.debug("Initiated multi-part upload for bucket '{}' key '{}' with id '{}'", new Object[]{S3OutputStream.this.bucket, S3OutputStream.this.key, uploadId});
        }

        public void uploadPart(ByteArrayInputStream inputStream, int partSize) {
            int currentPartNumber = this.partETags.size() + 1;
            UploadPartRequest request = (UploadPartRequest)new UploadPartRequest().withBucketName(S3OutputStream.this.bucket).withKey(S3OutputStream.this.key).withUploadId(this.uploadId).withSSECustomerKey(S3OutputStream.this.sseCustomerKey).withInputStream((InputStream)inputStream).withPartNumber(currentPartNumber).withPartSize((long)partSize).withGeneralProgressListener(S3OutputStream.this.progressListener);
            if (S3OutputStream.this.enableDigest) {
                request = request.withMD5Digest(this.computeDigest(inputStream, partSize));
            }
            log.debug("Uploading part {} for id '{}'", (Object)currentPartNumber, (Object)this.uploadId);
            this.partETags.add(S3OutputStream.this.s3.uploadPart(request).getPartETag());
        }

        private String computeDigest(ByteArrayInputStream inputStream, int partSize) {
            byte[] streamBytes = new byte[partSize];
            inputStream.read(streamBytes, 0, partSize);
            String digest = Base64.getEncoder().encodeToString(DigestUtils.md5((byte[])streamBytes));
            inputStream.reset();
            log.debug("Computed digest {} for id '{}'", (Object)digest, (Object)this.uploadId);
            return digest;
        }

        public void complete() throws IOException {
            log.debug("Completing multi-part upload for key '{}', id '{}'", (Object)S3OutputStream.this.key, (Object)this.uploadId);
            CompleteMultipartUploadRequest completeRequest = new CompleteMultipartUploadRequest(S3OutputStream.this.bucket, S3OutputStream.this.key, this.uploadId, this.partETags);
            S3OutputStream.handleAmazonExceptions(() -> S3OutputStream.this.s3.completeMultipartUpload(completeRequest));
        }

        public void abort() {
            log.warn("Aborting multi-part upload with id '{}'", (Object)this.uploadId);
            try {
                S3OutputStream.this.s3.abortMultipartUpload(new AbortMultipartUploadRequest(S3OutputStream.this.bucket, S3OutputStream.this.key, this.uploadId));
            }
            catch (Exception e) {
                log.warn("Unable to abort multipart upload, you may need to purge uploaded parts: ", (Throwable)e);
            }
        }
    }
}

